#include "MTM3GMViewer.h"
#include "MTAGLContext.h"
#include "MTApplication.h"
#include "MTFileList.h"
#include "MTTXMPViewer.h"
#include "MTExceptions.h"

#include <glut.h>

MTSimpleResourceTemplate	MTM3GMViewer::kM3GMFormatTemplate =
{
	{
		ONI_RESOURCE_HEADER
		
		{	kTypeUInt32 |
			kTypeIsFiller,	"Filler" },
		
		{	kTypeUInt32,	"PNTA ID",		kTypeInfoTypeID },	// 2
		{	kTypeUInt32,	"VCRA ID",		kTypeInfoTypeID },	// 3
		{	kTypeUInt32,	"VCRA ID",		kTypeInfoTypeID },	// 4
		{	kTypeUInt32,	"TXCA ID",		kTypeInfoTypeID },	// 5
		{	kTypeUInt32,	"IDXA ID",		kTypeInfoTypeID },	// 6
		{	kTypeUInt32,	"IDXA ID",		kTypeInfoTypeID },	// 7
		{	kTypeUInt32,	"TXMP ID",		kTypeInfoTypeID },	// 8
		
		{	kTypeUInt32,	"Unknown" },
		
		{	0 }
	}
};

MTM3GMViewer::MTM3GMViewer()
{
	pnta = nil;
	idxa = nil;
	txca = nil;
	textureData = nil;
	
	rotX =			0.0;
	rotY =			0.0;
	rotZ =			0.0;
	traX =			0.0;
	traY =			0.0;
	traZ =			-10.0;
	scale =			1.0;
	idxOffset =		0;
	useWireframe =	0;
	
	txmpIsGood =	0;
}

MTM3GMViewer::~MTM3GMViewer()
{
	delete pnta;
	delete idxa;
	delete txca;
	
	UnloadTexture();
}

void MTM3GMViewer::PostProcess(void)
{
	MTFileList				* fileList = GetOwningFileList();
	UInt32					idx;
	MTParsedResourceEntry	* entry;
	
	if(!pnta)
	{
		entry = GetResourceEntry(2);
		idx = fileList->theFile->LookupIdxFromID(entry->data);
		pnta = dynamic_cast<MTPNTAViewer *>(fileList->CreateViewerWindow(idx));
	}
	
	if(!idxa)
	{
		entry = GetResourceEntry(6);
		idx = fileList->theFile->LookupIdxFromID(entry->data);
		idxa = dynamic_cast<MTIDXAViewer *>(fileList->CreateViewerWindow(idx));
	}
	
	if(!txca)
	{
		entry = GetResourceEntry(5);
		idx = fileList->theFile->LookupIdxFromID(entry->data);
		txca = dynamic_cast<MTTXCAViewer *>(fileList->CreateViewerWindow(idx));
	}
	
	if(!txmpIsGood)
	{
		LoadTexture(GetResourceEntry(8)->data);
	}
}

void MTM3GMViewer::RecieveMessage(UInt32 messageType, UInt32 messageData)
{
	#pragma unused (messageData)
	
	switch(messageType)
	{
		case 'DONE':
			UpdateRequest();
			break;
		
		case 'EXPT':
			ExportAsPicture(nil);
			break;
		
		case 'INFO':
		{
			MTApplication::OpenResourceByIDRecord	record;
			
			gTheApp->DoMoveableModalDialog(1100, 1, 2, MTApplication::DoOpenResourceByIDItemProc, (long)&record);
			
			if(!record.canceled)
			{
				LoadTexture(record.id);
				
				UpdateRequest();
			}
		}
		break;
	}
}

UInt8 MTM3GMViewer::SupportsMessage(UInt32 messageType, UInt32 messageData)
{
	#pragma unused (messageData)
	
	switch(messageType)
	{
		case 'DONE':
		case 'EXPT':
		case 'INFO':
			return 1;
	}
	
	return 0;
}

void MTM3GMViewer::HandleKeyEvent(EventRecord * theEvent, UInt8 charCode, UInt8 keyCode, EventModifiers modifiers)
{
	#pragma unused (theEvent)
	#pragma unused (keyCode)
	
	UInt8		dirty = 0;
	//const float	rotateStep = 4;
	const float	rotateStep = 10;
	const float translateStep = .1;
	//const float scaleStep = .05;
	const float scaleStep = .5;
	
	if(modifiers & shiftKey)
	{
		switch(charCode)
		{
			case kLeftArrowCharCode:
				traX -= translateStep;
				dirty = 1;
				break;
			
			case kRightArrowCharCode:
				traX += translateStep;
				dirty = 1;
				break;
			
			case kUpArrowCharCode:
				traY -= translateStep;
				dirty = 1;
				break;
			
			case kDownArrowCharCode:
				traY += translateStep;
				dirty = 1;
				break;
			
			case ',':
			case '<':
				traZ -= translateStep * 10;
				dirty = 1;
				break;
			
			case '.':
			case '>':
				traZ += translateStep * 10;
				dirty = 1;
				break;
			
			case '=':
			case '+':
				scale += scaleStep;
				dirty = 1;
				break;
			
			case '-':
			case '_':
				scale -= scaleStep;
				dirty = 1;
				break;
			
			case 'w':
			case 'W':
				useWireframe ^= 1;
				dirty = 1;
				break;
		}
	}
	else
	{
		switch(charCode)
		{
			case kLeftArrowCharCode:
				rotX -= rotateStep;
				dirty = 1;
				break;
			
			case kRightArrowCharCode:
				rotX += rotateStep;
				dirty = 1;
				break;
			
			case kUpArrowCharCode:
				rotY -= rotateStep;
				dirty = 1;
				break;
			
			case kDownArrowCharCode:
				rotY += rotateStep;
				dirty = 1;
				break;
			
			case ',':
			case '<':
				rotZ -= rotateStep;
				dirty = 1;
				break;
			
			case '.':
			case '>':
				rotZ += rotateStep;
				dirty = 1;
				break;
			
			case '=':
			case '+':
				idxOffset++;
				dirty = 1;
				break;
			
			case '-':
			case '_':
				idxOffset--;
				dirty = 1;
				break;
			
			case 'w':
			case 'W':
				useWireframe ^= 1;
				dirty = 1;
				break;
		}
	}
	
	if(idxOffset < 0)
		idxOffset = 0;
	
	if(dirty)
		UpdateRequest();
}

void MTM3GMViewer::DrawToPort(GWorldPtr theWorld)
{
	const float		squareSize = 0.05;
	UInt8			isOpen = 0;
	UInt32			pointsIn = 0;
	UInt32			polygonsIn = 0;
	UInt32			txcaCounter = 0;
	
	if(gTheApp->hasAGL)
	{
		const std::vector <PNTAParsedEntry>	& localPointList = pnta->entries;
		const std::vector <UInt32>			& localIndexList = idxa->entries;
		std::vector <UInt8>					localPointStateList;
		
		localPointStateList.resize(localPointList.size());
		
		{
			GLfloat	lightPosition[4] = {0.0, 0.0, 5.0, 0.0};
			GLfloat	lightAmbient[4] = {0.2, 0.2, 0.2, 1.0};
			GLfloat lightDiffuse[4] = {0.5, 0.5, 0.5, 1.0};
			
			//glClearColor(0.0, 0.0, 0.0, 1.0);
			glClearColor(0.0, 0.0, 1.0, 1.0);
			glEnable(GL_DEPTH_TEST);
			glEnable(GL_NORMALIZE);
			glEnable(GL_AUTO_NORMAL);
			glLightfv(GL_LIGHT0, GL_AMBIENT, lightAmbient);
			glLightfv(GL_LIGHT0, GL_DIFFUSE, lightDiffuse);
			glLightfv(GL_LIGHT0, GL_POSITION, lightPosition);
			glEnable(GL_LIGHTING);
			glEnable(GL_LIGHT0);
			glShadeModel(GL_SMOOTH);
			//glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, 1);
			
			glMatrixMode(GL_PROJECTION);
			glLoadIdentity();
			glFrustum(-1.0, 1.0, -1.0, 1.0, 2.0, 100.0);
			
			glMatrixMode(GL_MODELVIEW);
			glLoadIdentity();
			glTranslatef(traX, traY, traZ);
			glRotatef(rotX, 1.0, 0.0, 0.0);
			glRotatef(rotY, 0.0, 1.0, 0.0);
			glRotatef(rotZ, 0.0, 0.0, 1.0);
		}
		
		if(textureData && txmpIsGood)
		{
			glGenTextures(1, &textureID);
			glBindTexture(GL_TEXTURE_2D, textureID);
			glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
			
			glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, txmp.width, txmp.height, 0, GL_RGBA, GL_UNSIGNED_BYTE, textureData);
			
			glEnable(GL_TEXTURE_2D);
			glBindTexture(GL_TEXTURE_2D, textureID);
			glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
		}
		
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		
		GLfloat blue[4] = {0.1, 0.1, 1.0, 0.2};
		GLfloat specular[4] = {0.4, 0.4, 0.4, 0.2};
		GLfloat	point1[3], point2[3], point3[3];
		
		//glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, blue);
		//glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular);
		
		glPushMatrix();
		
		for(UInt32 i = 0; i < localIndexList.size(); i++)
		{
			UInt32 data = localIndexList[i];
			
			if(data & 0x80000000)
			{
				if(isOpen)
				{
					glEnd();
					
					isOpen = 0;
				}
				
				if(useWireframe)
					glBegin(GL_LINE_STRIP);
				else
					glBegin(GL_TRIANGLE_STRIP);
				isOpen = 1;
				
				pointsIn = 0;
			}
			
			data &= 0x7FFFFFFF;
			
			data += 3;
			
			if(data >= localPointList.size())
				data = localPointList.size() - 1;
			
			txcaCounter = data - 3;
			if(txcaCounter >= txca->entries.size())
			{
				txcaCounter = 0;
			}
			
			if(textureData && txmpIsGood)
				glTexCoord2f(txca->entries[txcaCounter].x, txca->entries[txcaCounter].y);
			
			glVertex3f(localPointList[data].x, localPointList[data].y, localPointList[data].z);
			
			pointsIn++;
			
			for(UInt32 i = 0; i < 3; i++)
			{
				point3[i] = point2[i];
				point2[i] = point1[i];
			}
			
			point1[0] = localPointList[data].x;
			point1[1] = localPointList[data].y;
			point1[2] = localPointList[data].z;
			
			if(pointsIn < 3)
			{
				glNormal3f(0.0, 1.0, 0.0);
			}
			else
			{
				// very broken
				GLfloat	vector1[3], vector2[3], normal[3];
				
				if(pointsIn & 1)
				{
					for(UInt32 i = 0; i < 3; i++)
					{
						vector1[i] = point1[i] - point2[i];
						vector2[i] = point1[i] - point3[i];
					}
				}
				else
				{
					for(UInt32 i = 0; i < 3; i++)
					{
						vector1[i] = point2[i] - point1[i];
						vector2[i] = point2[i] - point3[i];
					}
				}
				
				normal[0] = (vector1[1] * vector2[2]) - (vector1[2] * vector2[1]);
				normal[1] = (vector1[2] * vector2[0]) - (vector1[0] * vector2[2]);
				normal[2] = (vector1[0] * vector2[1]) - (vector1[1] * vector2[0]);
				
				glNormal3fv(normal);
			}
			
			if(pointsIn >= 3)
			{
				polygonsIn++;
			}
		}
		
		if(isOpen)
		{
			glEnd();
		}
		
		glPopMatrix();
		
		if(0)
		{
			for(UInt32 i = 0; i < localPointStateList.size(); i++)
				localPointStateList[i] = 0;
			
			for(UInt32 i = 0; i < localIndexList.size(); i++)
			{
				UInt32 data = (localIndexList[i] & 0x7FFFFFF) + 3;
				
				if(idxOffset == i)
				{
					if(localIndexList[i] & 0x80000000)
					{
						localPointStateList[data] = 2;
					}
					else
					{
						localPointStateList[data] = 1;
					}
				}
			}
			
			for(UInt32 i = 0; i < localPointList.size(); i++)
			{
				glPushMatrix();
				
				switch(localPointStateList[i])
				{
					default:
					case 0:	glColor3d(1.0, 0.0, 0.0); break;
					case 1:	glColor3d(0.0, 1.0, 0.0); break;
					case 2:	glColor3d(0.0, 0.0, 1.0); break;
				}
				
				glTranslatef(localPointList[i].x, localPointList[i].y, localPointList[i].z);
				glutSolidCube(squareSize);
				
				glPopMatrix();
			}
		}
		
		glFlush();
		glFinish();
		
		CopyBits((BitMapPtr)*gTheAGLContext->offscreenWorld->portPixMap, (BitMapPtr)*theWorld->portPixMap, &gTheAGLContext->offscreenWorld->portRect, &theWorld->portRect, srcCopy, nil);
		
		{
			CGrafPtr	savePort;
			GDHandle	saveDevice;
			char		buf[4096];
			RGBColor	white = { 0xFFFF, 0xFFFF, 0xFFFF };
			
			GetGWorld(&savePort, &saveDevice);
			SetGWorld(theWorld, nil);
			
			RGBForeColor(&white);
			TextFont(4);
			TextSize(9);
			TextMode(srcOr);
			TextFace(0);
			//std::sprintf(	buf, "TX: %f TY: %f TZ: %f", traX, traY, traZ);
			std::sprintf(	buf, "polygons: %.8X vertices: %.8X %d %.8X", polygonsIn, pointsIn, txmpIsGood, txmpID);
			
			MoveTo(0, 12);
			DrawText(buf, 0, std::strlen(buf));
			
			SetGWorld(savePort, saveDevice);
		}
	}
	else
	{
		CGrafPtr	savePort;
		GDHandle	saveDevice;
		RGBColor	white = { 0xFFFF, 0xFFFF, 0xFFFF };
		RGBColor	black = { 0x0000, 0x0000, 0x0000 };
		
		GetGWorld(&savePort, &saveDevice);
		SetGWorld(theWorld, nil);
		
		RGBForeColor(&black);
		PaintRect(&theWorld->portRect);
		
		RGBForeColor(&white);
		TextFont(4);
		TextSize(9);
		TextMode(srcOr);
		TextFace(0);
		
		MoveTo(0, 12);
		DrawString("\pYou need OpenGL to see M3GM objects.");
		
		SetGWorld(savePort, saveDevice);
	}
	
	if(textureData && txmpIsGood)
	{
		glDeleteTextures(1, &textureID);
	}
}

void MTM3GMViewer::LoadTexture(UInt32 id)
{
	UInt8	* buf;
	UInt32	idx;
	
	UnloadTexture();
	
	txmpID = id;
	buf = GetOwningFileList()->theFile->LoadFileByID(id);
	idx = GetOwningFileList()->theFile->LookupIdxFromID(id);
	
	txmpIsGood = 0;
	
	if(buf)
	{
		try
		{
			txmp.AttachFileList(GetOwningFileList());
			txmp.ParseData(buf, GetOwningFileList()->theFile->fileList[idx].size);
			
			textureData = (UInt8 *)NewPtr(sizeof(UInt32) * txmp.width * txmp.height);
			if(!textureData)
				throw MTMemoryException("Out of memory creating texture");
			
			txmp.DrawToBuffer(textureData, sizeof(UInt32) * txmp.width, 32);
			
			// convert ARGB->RGBA
			{
				UInt32	* ptr = (UInt32 *)textureData;
				
				for(UInt32 i = 0; i < txmp.width * txmp.height; i++)
					ptr[i] = (ptr[i] << 8) | 0x000000FF;
			}
			
			txmpIsGood = 1;
		}
		catch(MTException err)
		{
			err.DoErrorDialog();
		}
		
		DisposePtr((Ptr)buf);
	}
}

void MTM3GMViewer::UnloadTexture(void)
{
	if(textureData)
	{
		DisposePtr((Ptr)textureData);
		textureData = nil;
	}
	
	txmpIsGood = 0;
}